/// @file deviceManagement.cpp
///
/// @brief Device management implementation
///
/// This file implements the DeviceInfo class.
/// This class stores device information and provide interface to check if a
/// device is of a certain type.
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see DeviceDetector Udev DD_EVENT_MASK

#include <algorithm>
#include <iostream>
#include <list>
#include <string>
#include <sstream>

#include "logger.h"
#include "deviceDetector.h"

namespace adit {
namespace uspi {

/// @var lDevInfo
///
/// The list of devices retrieved.
std::list<DeviceInfo> lDevInfo;

/* *************  USB definition  ************* */

/// @defgroup USB_HUB Definitions related to USB_HUB
/// This base class is defined for devices that are %USB hubs
/// and conform to the definition in the %USB specification.
///
/// @{
const int32_t USB_HUB_BASE_CLASS        = 0x09;   ///< 09h
const int32_t USB_HUB_SUB_CLASS         = 0x00;   ///< 00h
const int32_t USB_HUB_PROTOCOL_FS       = 0x00;   ///< 00h
const int32_t USB_HUB_PROTOCOL_HS_TT    = 0x01;   ///< 01h
const int32_t USB_HUB_PROTOCOL_HS_TTS   = 0x02;   ///< 02h
/// @}

/// @defgroup USB_MS Base class for Mass Storage devices
/// @{
const int32_t USB_MS_BASE_CLASS         = 0x08;    ///< 08h
/// @}


/// @defgroup USB_AOA Interface class to identify Android Accessory interfaces
/// @{
const int32_t USB_INTERFACE_CLASS_AOA       = 0xFF; ///< Interface class
const int32_t USB_INTERFACE_SUBCLASS_AOA    = 0xFF; ///< Interface subclass
const int32_t USB_INTERFACE_PROTOCOL_AOA    = 0x00; ///< Interface protocol
/// @}

/// @defgroup USB_DISCOVERY The USB discovery group
/// @{
/// @var USB_DISCOVERY_HOST_TO_DEVICE_TYPE
/// LIBUSB_ENDPOINT_IN (0x80) | LIBUSB_REQUEST_TYPE_VENDOR (0x02 << 5)
const int32_t USB_DISCOVERY_HOST_TO_DEVICE_TYPE = 0xC0;
/// Vendor request to get_protocol
const int32_t USB_DISCOVERY_GET_AOA_PROTOCOL    = 0x33;
/// @}

/* *************  Apple identifiers  ************* */

/// @defgroup USB_APPLE The USB Apple devices group
/// @{
const int32_t APPLE_VENDOR_ID               = 0x05AC; ///< Apple vendor ID
const int32_t APPLE_PRODUCT_ID_MIN          = 0x1200; ///< Apple minimum product ID
/// @}

/* *************  Unwired Technology identifier  ************* */
/// @defgroup UNWIRED_HUB The USB Unwired Technology group
/// @{
const int32_t UNWIRED_TECHNOLOGY_VENDOR_ID  = 0x2996; ///< Vendor ID
/// @}

/// @var dev_sysattr_str
/// List the strings to be used in udev get sys attributes and for
/// @ref DeviceInfo::dump
const char *dev_sysattr_str[] = {
    "ID_USB_INTERFACES",
    "serial",
    "product",
    "manufacturer",
    "address",
    "idVendor",
    "idProduct",
    "devpath",
    "devnum",
    "bDeviceClass",
    "bDeviceSubClass",
    "bDeviceProtocol",
    NULL
};

/// @brief Constructor, initializes @ref mMask to 0x0UL.
DeviceInfo::DeviceInfo() : mMask(0UL) { };

/// @brief Set a device info member
/// @param it The @ref dev_sysattr member's ID to be modified
/// @param in The input string to be used
///
/// If the @p it value is bigger or equal than the @ref DSYS_IDS, the integer
/// value corresponding to @p in will also be stored for convenience.
void DeviceInfo::setDeviceInfo(enum dev_sysattr it, const char *in)
{
    if (!in)
        return;

    if (it >= DSYS_IDS)
        std::istringstream(in) >> std::hex >> mDevInfo[it].miDevInfo;

    mDevInfo[it].msDevInfo.assign(in);
}

/// @brief Dumps the device info into the logger
///
/// Function used for debug purpose.
void DeviceInfo::dump()
{
    ddlog << kLogDebug << "Got Device" << std::endl;
    ddlog << kLogDebug << "   Node: " << mNode << std::endl;
    ddlog << kLogDebug << "   DevPath: " << mDevpath << std::endl;

    for (int i = DSYS_STRINGS; i < DSYS_NONE; ++i) {
        if (i >= DSYS_IDS)
            ddlog << kLogDebug << "   " << dev_sysattr_str[i] << ": 0x"
                << std::hex << mDevInfo[i].miDevInfo << std::endl;
        else
            ddlog << kLogDebug << "   " << dev_sysattr_str[i] << ": "
                << mDevInfo[i].msDevInfo << std::endl;
    }

    ddlog << kLogDebug << "   Mask: 0x" << std::hex << mMask << std::endl;
}

/// @brief Assignment operator
/// @param di The input DeviceInfo
///
/// Each members are copied into this.
DeviceInfo& DeviceInfo::operator=(const DeviceInfo &di)
{
    this->mNode = di.mNode;
    this->mDevpath = di.mDevpath;

    for (int i = DSYS_STRINGS; i < DSYS_NONE; ++i) {
        if (i >= DSYS_IDS)
            this->mDevInfo[i].miDevInfo = di.mDevInfo[i].miDevInfo;

        this->mDevInfo[i].msDevInfo = di.mDevInfo[i].msDevInfo;
    }

    this->mMask = di.mMask;

    return *this;
}

/// @brief Equality operator
/// @param di The input DeviceInfo to check for equality
///
/// This operator is used by list::find in order to find an object.
/// Therefore we do not look for strict equality here, as we are only
/// interested in finding elements with the same @ref mDevpath, expected to be
/// unique.
int DeviceInfo::operator==(const DeviceInfo &di) const
{
    return this->mDevpath == di.mDevpath;
}

/// @brief The different operator
/// @param di The DeviceInfo to check for difference
///
/// This operator is defined for compliance, as the only one used it the
/// DeviceInfo::operator==.
int DeviceInfo::operator!=(const DeviceInfo &di) const
{
    return !((*this) == di);
}

/// @brief The inferior operator
/// @param di The DeviceInfo to check for inferiority
///
/// This function is used by list::sort.
int DeviceInfo::operator<(const DeviceInfo &di) const
{
    return (this->mNode <= di.mNode) &&
        (this->mDevInfo[DSYS_ADDRESS].msDevInfo <=
         di.mDevInfo[DSYS_ADDRESS].msDevInfo);
}

/// @brief Check the device to setup its own @ref mMask
///
/// The method will first look for the device in the device list. If not
/// available, that means that the device has been removed. In this case, the
/// input device is return as is. If the device is found, the method will look
/// for several @ref mDevInfo in order to setup the @ref mMask.
/// @return The object containing the most information.
DeviceInfo& DeviceInfo::check()
{
    std::list<DeviceInfo>::iterator it = std::find(lDevInfo.begin(),
                                                   lDevInfo.end(),
                                                   *this);

    ddlog << kLogDebug << "Checking device" << std::endl;
    if (it == std::end(lDevInfo)) {
        ddlog << kLogInfo
            << "Device " << this->getNode() << " removed." << std::endl;
        return *this;
    }

    if ((*it).mDevInfo[DSYS_ADDRESS].msDevInfo != "") {
        ddlog << kLogDebug << "Got Bluetooth device." << std::endl;

        (*it).mMask |= DD_BLUETOOTH_OTHER;
        return (*it);
    }

    if (((*it).mDevInfo[DSYS_DEVICE_CLASS].miDevInfo == USB_HUB_BASE_CLASS) &&
        ((*it).mDevInfo[DSYS_DEVICE_SUB_CLASS].miDevInfo == USB_HUB_SUB_CLASS)) {
        ddlog << kLogDebug << "Got HUB." << std::endl;

        (*it).mMask |= DD_USB_HUB;
        return (*it);
    }

    switch ((*it).mDevInfo[DSYS_IDVENDOR].miDevInfo) {
    case UNWIRED_TECHNOLOGY_VENDOR_ID:
        ddlog << kLogDebug << "Got unwired tech device." << std::endl;

        (*it).mMask |= DD_USB_UNWIRED_TECHNOLOGY;
        break;
    case APPLE_VENDOR_ID:
        if ((*it).mDevInfo[DSYS_IDPRODUCT].miDevInfo >= APPLE_PRODUCT_ID_MIN) {
            ddlog << kLogDebug << "Got Apple device." << std::endl;

            (*it).mMask |= DD_USB_APPLE;
        } else {
            ddlog << kLogDebug << "Unknown Apple device." << std::endl;
        }
        break;
    default:
        ddlog << kLogDebug << "Got potential AOAP device." << std::endl;

        (*it).mMask |= DD_USB_AOAP;
        break;
    }

    return (*it);
}

} // namespace uspi
} // namespace adit
